
/* Copyright (c) 2008 Analog Devices, Inc. All rights reserved.

This source file shows how to interface to the AD7760/62/63/64/65 evaluation system
by calling the functions provided by ADI_CED1a.dll. (ADI_CED1a.dll is located
in the C:\WINDOWS\system32 folder once the evaluation software is installed.) 

Persons wishing to implement a program that interfaces to the AD7760/62/63/64/65 in C 
or C++ can use ADI_CED1_dll.h and ADI_CED1_dll.c to explicitly link to the DLL
and use this file as a reference.

Persons wishing to implement a program in another language should also use this
file as reference and link to the DLL in a way appropriate for that language.

Note the following files are necessary for ADI_CED1a.dll to function correctly
and are installed along with the dll during the evaluation software install

  C:\WINDOWS\system32\drivers\ADI_CED1.sys
  C:\WINDOWS\system32\drivers\ADI_CED1me.sys
  C:\WINDOWS\inf\ADI_CED1a.inf
  C:\WINDOWS\system32\ADI_CED1a\ADI_CED1a.spt
  C:\WINDOWS\system32\ADI_CED1a.hex

The path to file required to program the FPGA on the CED1 PCB must also be
passed in the form of a string to the program_board() function as shown below.

All paths given are given assuming that windows is installed in C:\WINDOWS
and may be different if this is not the case.

IMPORTANT: If evaluating another different AD776X change the #define AD7760 to 
AD776X 

POWER SUPPLIES: The Eval Board must be powered using a a power suply that 
applies a 7.5V between the V+ and GND terminals of Connector J2.

*/

#define AD7760   

#define FPGA_HEX_PATH "..\\CED1.hexout"

#define RES 24

#if defined AD7760
#define FSAMPLE 2.5    // 2.5 MSPS
#elif defined AD7762
#define FSAMPLE 0.5		// 0.625 MSPS
#elif defined AD7763 
#define FSAMPLE 0.5		// 0.625 MSPS
#endif 

#include<stdio.h>
#include<conio.h>
#include"ADI_CED1_dll.h"

void print_menu( void );

long init( ADI_CED1 ced, long *board_id );
long sample( ADI_CED1 ced, long board_id );
long writing( ADI_CED1 ced, long board_id );
long change_voltage( ADI_CED1 ced, long board_id );
//long gather_histogram( ADI_CED1 ced, long board_id );

int main( int argc, char *argv[] )
{
	long error;
	long menu_option;
	ADI_CED1 ced;
	long board_id;

	if( link_to_adi_ced1_dll( &ced ) )

	{
		printf( " Error linking to ADI_CED1a.dll \n" );
	}
	else
	{
		error = init( ced, &board_id );
		
		if( error )
			printf(" Error init %d\n", error ); 

		print_menu();
		
		while( '0' != ( menu_option = getch() ) )
		{

			switch( menu_option )
			{				
				case '1': /* Identify Board */
					error = ced.identify_board( board_id );
				break;
				
				case '2': /* Turn on Supplies */
					error = ced.configure_power_supplies( board_id, 2, 3, 3, 1 );
					/* This function controls the power supplies going from the 
					   CED1 PCB to the AD7760/62/63/64/65 evaluation PCB.
					   The above constants are fixed for supplies required by
					   the AD7760/62/63/64/65 evaluation board except the last parameter 
					   which is 1 to turn on the supplies and 0 to turn off 
					   the supplies. */
				break;
				
				case '3': /* Turn off Supplies */
					error = ced.configure_power_supplies( board_id, 2, 3, 3, 0 );					
					/* see comment in previous case */
				break;
				
				case '4': /* Sample */
					error = sample( ced, board_id );
				break;
				
				case '5': /* Gather Histogram */
				//	error = gather_histogram( ced, board_id );
					printf("Not avialable function");
				break;

				case '6': /* Reinitialise */
					error = init( ced, &board_id );
				break;

				case '7': /* Writing ADC Registers*/
					error = writing ( ced, board_id );
				break;

				case '8': /* Change voltages*/
					error = change_voltage ( ced, board_id );
				break;

			}

			if( error )
				printf(" Error %d\n", error ); 
		}
	}
	free_adi_ced1_dll( ced );

	return 0;
}


long change_voltage( ADI_CED1 ced, long board_id )
{
	long error;

	unsigned short volt_array[2];
	long size_of_volt_array = 2;

	int contador = 0;

	volt_array[0] = 0x9330;
	volt_array[1] = 0x8000;

	error = ced.manual_send_to_ced( board_id, 0x0301, volt_array,
			size_of_volt_array );

	/*while ( contador < 512)
	{
		contador ++;

		volt_array[0] ++;
		
		error = ced.manual_send_to_ced( board_id, 0x0301, volt_array,
			size_of_volt_array );

		printf ("%d", contador);

	}*/
	
	
	
	return error;

}


void print_menu( void )
{
	printf( "                     \n");
	printf( " 1 Identify Board    \n" );
	printf( " 2 Turn on Supplies  \n" );
	printf( " 3 Turn off Supplies \n" );
	printf( " 4 Sample            \n" );
	printf( " 5 Gather Histogram  \n" );
	printf( " 6 Reinitialise      \n" );
	printf( " 7 Writing ADC Registers \n" );
	printf( " 8 Change Voltage \n" );
	printf( " 0 Exit              \n" );
	printf( "                     \n" );
}

long writing ( ADI_CED1 ced, long board_id )
{
	long error;

	unsigned short spi_array[2];
	long size_of_spi_array = 2;


	/*************************************************************************/
	/*************************CONFIGURAR EL ADC*******************************/

	/* configure the software selection of ADC 
	   the AD7760/62/63/64/65 has two registers CR1 and CR2 (see data sheet for more 
	   details). These pins configure the control register and are connected
	   to the spi pins on the CED1 PCB as follows:
	   
	
		--------------------
		 Control Register 2:
		--------------------
		// Frame[5:0] 5 bits for Control Register 2
		// If Frame[11] == 1 => Send Control Register 2
		Frame[10] = 0 -> CR2 
		Frame[10] = 1 -> CR1

				/*--------------------------
				  D5 D4 D3  D2 D1 D0 
				---------------------
				nCDIV0  PD LPWR 1D1PD
				--------------------------*/

		// The header of the frame is 16'b1111_XX11_11XX_XXXX

	/*
	Frame[5:0] = 6'b000010
	Frame[10] = 1'b0
	Frame[11] = 1'b1

	CR2: 1111_1011_1100_0010
	*/   
                                
	spi_array[0] = 0xFBC2;		// Normal power 
	//spi_array[0] = 0xFBC6;     // Low power
	spi_array[1] = 0x0000;

	error = ced.manual_send_to_ced( board_id, 0x0030, spi_array,
			size_of_spi_array );


	/*  -------------------------
	    Control Register 1 (CR1):
	    -------------------------

		// Frame[9:0] 10 bits for Control Register 1
		// If Frame[11] == 1 => Send Control Register 1
		Frame[10] = 0 -> CR2 
		Frame[10] = 1 -> CR1

	/*---------------------------------------------------------
	 D9  D8   D7   D6    D5   D4    D3   D2  D1  D0             
	------------------------------------------------	
	SYNCFLEN3FLEN2FLEN1FLEN0nBYPF3nBYPF1DEC2DEC1DEC0
	---------------------------------------------------------*/

	// The header of the frame is 16'b1111_XXXX_XXXX_XXXX

	/*
	Frame[9:0] = 5'b0000000000
	Frame[10] = 1'b1
	Frame[11] = 1'b1

	CR2: 1111_1100_0001_1010
	*/

	spi_array[0] = 0xFC18; //decimation rate 16
	spi_array[1] = 0x0000;

	if( !error )
		error = ced.manual_send_to_ced( board_id, 0x0030, spi_array,
			size_of_spi_array );

	/*************************************************************************/



	if ( error )
		printf(" Writing registers error %d \n", error );


	printf (" Escritura Realizada \n");

	return error;
	
}



long init( ADI_CED1 ced, long *board_id )
{
	long error;
	long number_boards;
	unsigned long board_ids[8];
	const char *hexout_path = FPGA_HEX_PATH;

	/* get board id */
	error = ced.search_for_boards( &number_boards, board_ids );
	/* use id of first board found */
	*board_id = board_ids[0];
	
	printf(" board_id %d \n", board_id );

	/* program the FPGA on the board */
	if( !error ){
		error = ced.program_board( *board_id, hexout_path );		
	}
	else
		printf(" Error search board %d \n", error );

	/* Turn on Supplies */
	/* This function controls the power supplies going from the 
	   CED1 PCB to the AD7760/62/63/64/65 evaluation PCB.
	   The above constants are fixed for supplies required by
	   the AD7760/62/63/64/65 evaluation board except the last parameter 
	   which is 1 to turn on the supplies and 0 to turn off 
	   the supplies. */
	if( !error )
		error = ced.configure_power_supplies( *board_id, 2, 3, 3, 1 );
	else
		printf(" error program board %d \n", error );
	

	if ( error )
		printf(" Error configure %d \n", error );

	

	return error;
}

long sample( ADI_CED1 ced, long board_id )
{
	long error;
	unsigned short send_array[2];       /* fixed size for AD7760/62/63/64/65 eval */
	long size_of_send_array = 2;        /* fixed size for AD7760/62/63/64/65 eval */
	long size_of_samples_array = 200; /* max: 2097152 min: 2, must be even */ 
	long size_of_samples_array_rec = 400;	
	short samples_array_rec[400];
	
	long compound_samples[200];
	long compound_samples2;

	long counter = 0;

	long data_ready = FALSE;  
	int i;
	

	FILE *dump_file;


	/* ** send request to start sampling ** */
	send_array[0] = (short)(size_of_samples_array >> 16);
	send_array[1] = (short)size_of_samples_array;

	error = ced.manual_send_to_ced( board_id, 0x0010, send_array,
		size_of_send_array );
	 
	/* ** check to see if samples are collected before upload ** */
	while( !data_ready && !error )
		error = ced.check_data_ready( board_id, &data_ready );
	/* it may be necessary to include timing and a time out exit condition
	   in the above loop */

	
	/* ** upload samples ** */
	if( !error )
		error = ced.manual_receive_from_ced( board_id, 0x0010,
			samples_array_rec, size_of_samples_array_rec );

	
	for( i=0; i < size_of_samples_array_rec; i += 2 )
	{
		counter = counter + 1;

		compound_samples[counter] = (long) samples_array_rec[i];
		
		if ( counter == 1 )
			printf("%d \n", compound_samples[counter]);


		compound_samples[counter] = compound_samples[counter] << 8;

		if ( counter == 1 )
			printf("%d \n", compound_samples[counter]);


		compound_samples2 = 0x0000FF00 & ((long) samples_array_rec[i+1]);  //0000FF

		if ( counter == 1 )
			printf("%d \n", compound_samples2);

		compound_samples2 = compound_samples2 >> 8;

		if ( counter == 1 )
			printf("%d \n", compound_samples2);

		compound_samples[counter] = compound_samples[counter] | compound_samples2;

		if ( counter == 1 )
			printf("%d \n", compound_samples[counter]);

	}



	for( i=0; i < size_of_samples_array; i++ )
		if( compound_samples[i] >> (RES-1) )
			compound_samples[i] |= 0xFFFFUL << RES;

 

	/* eg dump data to file */
	dump_file = fopen( "samples.csv", "w" );

	if( dump_file != NULL )
	{
		for( i = 0; i < size_of_samples_array; i += 1 )
			fprintf( dump_file, "%d \n", compound_samples[i]);		
			
		fclose( dump_file );
	}


	return error;
}


	